//
//  EtasRegister.h
//  etasSdk
//
//  Created by ReidLee on 2018/6/15.
//  Copyright © 2018年 esandinfo. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "EtasResult.h"
#import "EtasConfiguration.h"
#import "IfaaCommon.h"

/**
 * 注册类, 具体的使用，可参考 [IFAA 注册](http://bbs.esandinfo.com/topic/44/ifaa流程拆分-ios-ifaa-注册)
 */
@interface EtasRegister : NSObject

/**
 * 初始化函数
 - param
    - config IFAA 执行上下文 `EtasConfiguration`
 */
- (id)initWithEtasConfiguration:(EtasConfiguration *) config;

/**
 * 注册初始化
 - return
     <table border="1">
         <tr>
             <th>可能的错误码</th>
             <th>内容</th>
             <th>备注</th>
         </tr>
         <tr>
             <td>IFAA_STATUS_PASSCODE_NOT_SET</td>
             <td>错误描述</td>
             <td>锁屏密码没设置</td>
         </tr>
         <tr>
             <td>IFAA_STATUS_NOT_ENROLLED</td>
             <td>错误描述</td>
             <td>设备尚未录入任何指纹，此处可引导用户跳转到指纹界面录入指纹后再做注册处理</td>
         </tr>
         <tr>
             <td>IFAA_SUCCESS</td>
             <td>注册请求的报文数据</td>
             <td>此数据需要发送到bizServer服务器，请求注册</td>
         </tr>
     </table>
 */
- (EtasResult *) regInit;

/**
 * 执行注册操作
 - param
    - msg 服务器返回的报文数据
    - etasCallback  回调的 EtasResult 参数可能有如下内容 `EtasCallback`
     <table border="1">
         <tr>
             <th>可能的错误码</th>
             <th>内容</th>
             <th>备注</th>
         </tr>
         <tr>
             <td>IFAA_SERVER_ERROR</td>
             <td>错误描述</td>
             <td>返回的报文数据异常</td>
         </tr>
         <tr>
             <td>IFAA_POLICY_REJECTED</td>
             <td>错误描述 </td>
             <td>IFAA 已被策略管理禁用</td>
         </tr>
         <tr>
             <td>IFAA_USER_REJECTED</td>
             <td>错误描述</td>
             <td>用户已被禁用</td>
         </tr>
         <tr>
             <td>IFAA_APPID_NOT_FOUND</td>
             <td>错误描述</td>
             <td>应用白名单中未设置应用标识</td>
         </tr>
         <tr>
             <td>IFAA_DEVICE_MODEL_NOT_FOUND</td>
             <td>错误描述</td>
             <td>设备型号不存在</td>
         </tr>
         <tr>
             <td>IFAA_SIGNATURE_FAIL</td>
             <td>错误描述</td>
             <td>未获取到签名数据</td>
         </tr>
         <tr>
             <td>IFAA_STATUS_RESULT_CANCELED</td>
             <td>错误描述</td>
             <td>用户取消</td>
         </tr>
         <tr>
             <td>IFAA_STATUS_RESULT_AUTH_FAIL</td>
             <td>错误描述</td>
             <td>验证失败，系统指纹不匹配</td>
         </tr>
         <tr>
             <td>IFAA_STATUS_RESULT_FALLBACK</td>
             <td>错误描述</td>
             <td>点击了 FALLBACK 按钮</td>
         </tr>
         <tr>
             <td>IFAA_PERMISSION_DENIED</td>
             <td>错误描述</td>
             <td>当前应用未获取Face ID权限</td>
         </tr>
         <tr>
             <td>IFAA_AUTHENTICATOR_DISABLE</td>
             <td>错误描述</td>
             <td>认证器被禁用</td>
         </tr>
         <tr>
             <td>IFAA_AUTHENTICATOR_NOT_FOUND</td>
             <td>错误描述</td>
             <td>未找到相应的认证器</td>
         </tr>
         <tr>
             <td>IFAA_DEVICE_KEY_NOT_FOUND</td>
             <td>错误描述</td>
             <td>设备密钥被禁用</td>
         </tr>
         <tr>
             <td>IFAA_PROTECT_PAYLOAD_NOT_MATCH</td>
             <td>错误描述</td>
             <td>受保护的业务数据不匹配</td>
         </tr>
         <tr>
             <td>IFAA_CLIENT_ERROR</td>
             <td>异常信息</td>
             <td>异常信息有如下几种
                 <ul>
                     <li>RESULT_NOT_SUPPORT</li>
                     <li>RESULT_SYSTEM_CANCEL</li>
                     <li>RESULT_PASSCODE_NOTSET</li>
                     <li>RESULT_NOT_AVAILABLE</li>
                     <li>RESULT_TOUCHID_ERRO</li>
                     <li>RESULT_JAILBREAK</li>
                     <li>RESULT_WRONG_PARAM</li>
                     <li>RESULT_LOCKOUT</li>
                     <li>RESULT_UNKNOWN</li>
                 </ul>
             </td>
             </tr>
         <tr>
             <td>IFAA_SUCCESS</td>
             <td>需要同步到服务器的报文数据 </td>
             <td>数据经过签名，可保证不会被篡改</td>
         </tr>
     </table>

 ```c
 typedef enum {

     RESULT_NOT_SUPPORT,         //系统版本低于iOS8
     RESULT_SYSTEM_CANCEL,       //被其他进程抢占，如来电
     RESULT_PASSCODE_NOTSET,     //passcode未设置
     RESULT_NOT_ENROLLED,        //未录入指纹
     RESULT_TOUCHID_ERRO,        //系统指纹异常
     RESULT_JAILBREAK,           //该机器越狱 (此状态码不可用)
     RESULT_WRONG_PARAM,         //参数不对
     RESULT_LOCKOUT,             //其他异常
     RESULT_UNKNOWN,             //其他异常
 } IFAABioResult;
 ```
 */
- (void) register:(NSString *)msg etasCallback:(EtasCallback) etasCallback;

/**
 * 结束注册操作
 - param
    - msg 服务器返回的注册报文
 - eturn
     <table border="1">
         <tr>
             <th>可能的错误码</th>
             <th>内容</th>
             <th>备注</th>
         </tr>
         <tr>
             <td>IFAA_SERVER_ERROR</td>
             <td>错误描述</td>
             <td>解析服务器返回的报文发生异常</td>
         </tr>
         <tr>
             <td>IFAA_POLICY_REJECTED</td>
             <td>错误描述 </td>
             <td>IFAA 已被策略管理禁用</td>
         </tr>
         <tr>
             <td>IFAA_USER_REJECTED</td>
             <td>错误描述</td>
             <td>用户已被禁用</td>
         </tr>
         <tr>
             <td>IFAA_APPID_NOT_FOUND</td>
             <td>错误描述</td>
             <td>应用白名单中未设置应用标识</td>
         </tr>
         <tr>
             <td>IFAA_DEVICE_MODEL_NOT_FOUND</td>
             <td>错误描述</td>
             <td>设备型号不存在</td>
         </tr>
         <tr>
             <td>IFAA_SIGNATURE_FAIL</td>
             <td>错误描述</td>
             <td>未获取到签名数据</td>
         </tr>
         <tr>
             <td>IFAA_SUCCESS</td>
             <td>注册的 token</td>
             <td>表明注册成功了</td>
         </tr>
     </table>
 */
- (EtasResult *) regFinish:(NSString *) msg;

/*
 设置密码(当认证类型为pin/手势时候调用)
    @param passWord 密码数据，可以是密码的摘要
    @param isStore 指定是否在本地存储密码/手势加盐后的签名加密后的数据。
    @return 执行结果
 */
- (BOOL) setPassword: (NSString*) passWord isStore: (BOOL) isStore;

@end
