//
//  EtasVerify.h
//  etasSdk
//
//  Created by ReidLee on 2018/6/15.
//  Copyright © 2018年 esandinfo. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "EtasResult.h"
#import "EtasConfiguration.h"
#import "IfaaCommon.h"

/**
 * IFAA 认证类, 关于此类的使用，可参考 [IFAA 认证](http://bbs.esandinfo.com/topic/45/ifaa流程拆分-ios-ifaa认证)
 */
@interface EtasVerify : NSObject

// 指位更新报文数据，暂不对外公开
@property  (nonatomic, copy) NSString *ifaaTemplateUdMessage;

/**
 * 初始化函数
 - param `EtasConfiguration`
    - config IFAA 执行上下文 `EtasConfiguration`
 */
- (id)initWithEtasConfiguration:(EtasConfiguration *) config;

/**
 * 认证初始化
 - return `EtasResult` 返回的状态码可参考 `IFAAErrorCodeEnum`
     <table border="1">
         <tr>
             <th>可能的错误码</th>
             <th>内容</th>
             <th>备注</th>
         </tr>
         <tr>
             <td>`IFAA_STATUS_PASSCODE_NOT_SET`</td>
             <td>错误描述</td>
             <td>锁屏密码没设置</td>
         </tr>
         <tr>
             <td>`STATUS_NOT_ENROLLED`</td>
             <td>错误描述</td>
             <td>设备尚未录入任何指纹，此处可引导用户跳转到指纹界面录入指纹后再做认证处理</td>
         </tr>
         <tr>
             <td>`IFAA_STATUS_NOT_REGISTERED`</td>
             <td>错误描述</td>
             <td>ifaa 尚未注册，不能执行认证操作</td>
         </tr>
         <tr>
             <td>`IFAA_SUCCESS`</td>
             <td>认证请求的报文</td>
             <td>需要把报文同步给服务器</td>
         </tr>
     </table>
 */
- (EtasResult *) authInit;

/**
 * 执行认证操作

 - param
    - msg  认证请求报文
    - etasCallback 回调的 EtasResult 参数可能有如下内容 `EtasCallback` , 关于IFAA_CLIENT_ERROR 错误的处理，可参考 [关于ifaa认证过程中出现的 IFAA_CLIENT_ERROR 处理](http://bbs.esandinfo.com/topic/53/ifaa流程拆分-ios-关于ifaa认证过程中出现的-ifaa_client_error-处理)
     <table border="1">
         <tr>
             <th>可能的错误码</th>
             <th>内容</th>
             <th>备注</th>
         </tr>
         <tr>
             <td>IFAA_SERVER_ERROR</td>
             <td>错误描述</td>
             <td>返回的报文数据异常</td>
         </tr>
         <tr>
             <td>IFAA_POLICY_REJECTED</td>
             <td>错误描述 </td>
             <td>IFAA 已被策略管理禁用</td>
         </tr>
         <tr>
             <td>IFAA_USER_REJECTED</td>
             <td>错误描述</td>
             <td>用户已被禁用</td>
         </tr>
         <tr>
             <td>IFAA_APPID_NOT_FOUND</td>
             <td>错误描述</td>
             <td>应用白名单中未设置应用标识</td>
         </tr>
         <tr>
             <td>IFAA_DEVICE_MODEL_NOT_FOUND</td>
             <td>错误描述</td>
             <td>设备型号不存在</td>
         </tr>
         <tr>
             <td>IFAA_SIGNATURE_FAIL</td>
             <td>错误描述</td>
             <td>未获取到签名数据</td>
         </tr>
         <tr>
             <td>IFAA_STATUS_RESULT_CANCELED</td>
             <td>错误描述</td>
             <td>用户取消</td>
         </tr>
         <tr>
             <td>IFAA_STATUS_RESULT_AUTH_FAIL</td>
             <td>错误描述</td>
             <td>验证失败，系统指纹不匹配</td>
         </tr>
         <tr>
             <td>IFAA_STATUS_RESULT_FALLBACK</td>
             <td>错误描述</td>
             <td>点击了 FALLBACK 按钮</td>
         </tr>
         <tr>
             <td>IFAA_PERMISSION_DENIED</td>
             <td>错误描述</td>
             <td>当前应用未获取Face ID权限</td>
         </tr>
         <tr>
             <td>IFAA_AUTHENTICATOR_DISABLE</td>
             <td>错误描述</td>
             <td>认证器被禁用</td>
         </tr>
         <tr>
             <td>IFAA_AUTHENTICATOR_NOT_FOUND</td>
             <td>错误描述</td>
             <td>未找到相应的认证器</td>
         </tr>
         <tr>
             <td>IFAA_DEVICE_KEY_NOT_FOUND</td>
             <td>错误描述</td>
             <td>设备密钥被禁用</td>
         </tr>
         <tr>
             <td>IFAA_PROTECT_PAYLOAD_NOT_MATCH</td>
             <td>错误描述</td>
             <td>受保护的业务数据不匹配</td>
         </tr>
         <tr>
             <td>IFAA_CLIENT_ERROR</td>
             <td>异常信息</td>
             <td>异常信息有如下几种
                <ul>
                    <li>RESULT_USER_CANCEL</li>
                     <li>RESULT_NOT_SUPPORT</li>
                     <li>RESULT_SYSTEM_CANCEL</li>
                     <li>RESULT_PASSCODE_NOTSET</li>
                     <li>RESULT_NO_MATCH</li>
                     <li>RESULT_FALLBACK</li>
                      <li>RESULT_NOT_AVAILABLE</li>
                      <li>RESULT_TOUCHID_ERRO</li>
                      <li>RESULT_JAILBREAK</li>
                      <li>RESULT_WRONG_PARAM</li>
                      <li>RESULT_LOCKOUT</li>
                      <li>RESULT_UNKNOWN</li>
                </ul>
            </td>
         </tr>
         <tr>
             <td>IFAA_SUCCESS</td>
             <td>需要同步到服务器的报文数据</td>
             <td>数据经过签名，可保证不会被篡改</td>
         </tr>
     </table>

 ```c
 typedef enum {

     RESULT_USER_CANCEL,         //用户点击了'cancel'
     RESULT_NOT_SUPPORT,         //系统版本低于iOS8
     RESULT_SYSTEM_CANCEL,       //被其他进程抢占，如来电
     RESULT_PASSCODE_NOTSET,     //passcode未设置
     RESULT_NO_MATCH,            //指纹不匹配
     RESULT_FALLBACK,            //用户点击了'Enter Passcode'
     RESULT_NOT_ENROLLED,        //未录入指纹
     RESULT_NOT_AVAILABLE,       //不可用
     RESULT_TOUCHID_ERRO,        //系统指纹异常
     RESULT_JAILBREAK,           //该机器越狱 (此状态码不可用)
     RESULT_WRONG_PARAM,         //参数不对
     RESULT_LOCKOUT,             //连续多次失败，被锁定(3次)
     RESULT_UNKNOWN,             //其他异常

 } IFAABioResult;
 ```
 */
- (void) auth:(NSString *)msg etasCallback:(EtasCallback) etasCallback;

/**
 * 对业务数据进行验证
 * @param protectPayload 业务数据报文
 * @return
 */
- (bool) verifyProtectPayload:(NSString *) protectPayload;

/**
 * 通知sdk 结束认证，并返回认证结果
- param ifaaMsg 服务器返回的认证报文
- return `EtasResult`
 <table border="1">
     <tr>
         <th>可能的错误码</th>
         <th>内容</th>
         <th>备注</th>
     </tr>
     <tr>
         <td>IFAA_SERVER_ERROR</td>
         <td>错误描述</td>
         <td>解析服务器返回的报文发生异常</td>
     </tr>
     <tr>
         <td>IFAA_POLICY_REJECTED</td>
         <td>错误描述 </td>
         <td>IFAA 已被策略管理禁用</td>
     </tr>
     <tr>
         <td>IFAA_USER_REJECTED</td>
         <td>错误描述</td>
         <td>用户已被禁用</td>
     </tr>
     <tr>
         <td>IFAA_APPID_NOT_FOUND</td>
         <td>错误描述</td>
         <td>应用白名单中未设置应用标识</td>
     </tr>
     <tr>
         <td>IFAA_DEVICE_MODEL_NOT_FOUND</td>
         <td>错误描述</td>
         <td>设备型号不存在</td>
     </tr>
     <tr>
         <td>IFAA_SIGNATURE_FAIL</td>
         <td>错误描述</td>
         <td>未获取到签名数据</td>
     </tr>
     <tr>
         <td>IFAA_WRONG_AUTHDATAINDEX</td>
         <td>不匹配的指位的报文信息</td>
         <td>此时可把指位信息更新到服务器上，下次此指位即可使用，</td>
     </tr>
     <tr>
         <td>IFAA_SUCCESS</td>
         <td>注册的 token</td>
         <td>表明认证成功了</td>
     </tr>
 </table>
 */
- (EtasResult *) authFinish:(NSString *) msg;

/*
 本地校验 (当认证类型为手势/PIN的时候，并且注册时候不在本地保存注册数据时候使用)
 @param passWord 密码明文，或者密码摘要
 @return 校验结果
 */
- (Boolean) verifyPassword: (NSString*) passWord;

/*
 设置手势/PIN数据 (当认证类型为手势/PIN的时，并且注册时候不在本地保存注册数据时候使用)
 @param passWord 密码明文，或者密码摘要
 @return 处理结果
 */
- (BOOL) setPassword: (NSString*) passWord;

@end
